/*
 * --COPYRIGHT--,BSD Copyright (c) 2015, Texas Instruments Incorporated
 * All rights reserved. Redistribution and use in source and binary
 * forms, with or without modification, are permitted provided that the
 * following conditions are met: * Redistributions of source code must
 * retain the above copyright notice, this list of conditions and the
 * following disclaimer. * Redistributions in binary form must reproduce
 * the above copyright notice, this list of conditions and the following
 * disclaimer in the documentation and/or other materials provided with
 * the distribution. * Neither the name of Texas Instruments Incorporated 
 * nor the names of its contributors may be used to endorse or promote
 * products derived from this software without specific prior written
 * permission. THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE. --/COPYRIGHT--
 */

#include "msp430f2617.h"
#include "utility.h"
#include "Serial_Cmd_Monitor.h"
#include "uart.h"

/*****************************************************************************/

boolean         force_update = false;

// variables to calculate ifs
unsigned int    temp_pwm_ain1 = 1000;	// set out of range, force
					// recalculation
unsigned int    temp_pwm_ain2 = 1000;	// set out of range, force
					// recalculation
unsigned int    temp_pwm_bin1 = 1000;	// set out of range, force
					// recalculation
unsigned int    temp_pwm_bin2 = 1000;	// set out of range, force
					// recalculation
unsigned int    temp_avref_val = 32000;	// set out of range, force
					// recalculation
unsigned int    temp_bvref_val = 32000;	// set out of range, force
					// recalculation
unsigned int    temp_G_Mode = 100;	// set out of range, force
					// recalculation
unsigned int    temp_G_RUN = 100;	// set out of range, force
					// recalculation
unsigned int    temp_G_DIRECTION = 100;	// set out of range, force
					// recalculation
unsigned int    temp_G_PPS_Value = 12000;	// set out of range, force 
						// recalculation
unsigned int    temp_torque_val = 500;	// set out of range, force
					// recalculation
unsigned int    temp_toff = 32000;	// set out of range, force
					// recalculation
unsigned int    temp_adecay = 32000;	// set out of range, force
					// recalculation
unsigned int    temp_bdecay = 32000;	// set out of range, force
					// recalculation


float           temp_G_IRSENSE = 100;	// set out of range, force
					// recalculation

// PWM Generation
unsigned int    stepper_running = 0;	// flag signals stepper is running
unsigned int    Timer_Value;	// frequency setting in stepper motor

int             i;
int             flag;



/*****************************************************************************/

// Function Definitions

// Initialization of clocks and Ports
void
Initialize()
{
    // Setup clocks
    // set main clock to 8MHz
    DCOCTL = CALDCO_8MHZ;	// Set DCO to 8MHz
    BCSCTL1 = CALBC1_8MHZ | XTS;
    // no effect
    BCSCTL3 = LFXT1S_2;		// ACLK = VLO
    // select 8MHZ DCO for MCLK,
    // 4MHz for SMCLK
    BCSCTL2 = DIVS_1;



    WDTCTL = WDTPW | WDTHOLD;	// Stop watchdog timer
    // Configure port directions and peripherals as needed
    // Note: All unused ports should be set to either output or input with
    // pullup/down enabled

    // 
    P1DIR = BIT7 | BIT6 | BIT5 | BIT4 | BIT3 | BIT2 | BIT1 | BIT0;	// All 
									// high
    P1OUT = BIT7 | BIT6 | BIT5 | BIT4 | BIT3 | BIT2 | BIT1 | BIT0;	// All 
									// high

    P3DIR = BIT7 | BIT6 | BIT4 | BIT3 | BIT2 | BIT1 | BIT0;	// All
								// high
								// except
								// bit 5
								// (RXD)
    P3OUT = BIT7 | BIT6 | BIT4 | BIT3 | BIT2 | BIT1 | BIT0;	// All
								// high

    P5DIR = BIT7 | BIT6 | BIT5 | BIT4 | BIT3 | BIT2 | BIT1 | BIT0;	// All 
									// high
    P5OUT = BIT7 | BIT6 | BIT5 | BIT4 | BIT3 | BIT2 | BIT1 | BIT0;	// All 
									// high

    P6DIR = BIT7 | BIT6 | BIT5 | BIT4 | BIT3 | BIT2 | BIT1 | BIT0;	// All 
									// high
    P6OUT = BIT7 | BIT6 | BIT4 | BIT3 | BIT2 | BIT1 | BIT0;	// All
								// high
								// except
								// bit 5

    P7DIR = BIT7 | BIT6 | BIT5 | BIT4 | BIT3 | BIT2 | BIT1 | BIT0;	// All 
									// high
    P7OUT = BIT7 | BIT6 | BIT5 | BIT4 | BIT3 | BIT2 | BIT1 | BIT0;	// All 
									// high

    P8DIR = BIT7 | BIT6 | BIT5 | BIT4 | BIT3 | BIT2 | BIT1 | BIT0;	// All 
									// high
    P8OUT = BIT7 | BIT6 | BIT5 | BIT4 | BIT3 | BIT2 | BIT1 | BIT0;	// All 
									// high

    PADIR = BIT7 | BIT6 | BIT5 | BIT4 | BIT3 | BIT2 | BIT1 | BIT0;	// All 
									// high
    PAOUT = BIT7 | BIT6 | BIT5 | BIT4 | BIT3 | BIT2 | BIT1 | BIT0;	// All 
									// high

    P2DIR = TOFF | PARA | TRQ0 | AIN2 | AIN1 | TRQ1 | ADECAY | BDECAY;	// Bits 
									// 6:0
    P4DIR = WAKE | BIT4 | BIT3 | BIN1 | BIN2 | BIT0;	// FAULT is input
    enable_V3P3_GPIO;

    // Configure DACs 0 and 1
    DAC12_0CTL = DAC12SREF_3 | DAC12IR | DAC12AMP_7;	// Select VeREF,
							// 1x, and Amp
							// high speed/high 
							// current
    DAC12_0CTL &= ~DAC12OPS;

    DAC12_1CTL = DAC12SREF_3 | DAC12IR | DAC12AMP_7;	// Select VeREF,
							// 1x, and Amp
							// high speed/high 
							// current
    DAC12_1CTL &= ~DAC12OPS;

    // UART Initialization
    uartInit();

    // Enables LPM Interrupts
    __bis_SR_register(GIE);

    // GUI Composer Monitor Initialization
    ClearBufferRelatedParam();

    // Set Default GUI Variables Values
    SetDefaultGUIVals();

    // Set Default GPIO Values
    SetDefaultGPIOVals();

    // Load System Values
    UpdateSystem();

}				// end of Initialize

// This routine:
// Reads GUI variables to determine device mode
// Sets the inputs to proper levels (PWM, static, float)
// Minimizes updates to reduce mcu requirements
// Forces an update every 10000 times through to ensure
// sync
void
UpdateSystem()
{
    LED_TOGGLE_CNT++;
    // toggle status LED if count is reached
    if (LED_TOGGLE_CNT == LED_TIMEOUT) {
	toggle_status_led;
	LED_TOGGLE_CNT = 0;
	force_update = true;	// force an update of the chopping current
    }
    // From GUI to Micro
    // Update the GPIO Pins from Global Variables
    // WAKE
    if (G_WAKE == high)
	set_WAKE_hi;		// exit sleep mode, capture mode used
    else
	set_WAKE_lo;		// enter sleep mode

    // Mode selection
    // only update mode when a change is detected
    if (temp_torque_val != G_TORQUE || force_update == true) {
	// TORQUE CONTROL
	// TORQUE CONTROL -- TRQ1 CNTL
	switch (G_TORQUE) {
	case 0:		// 00
	    set_TRQ1_lo;
	    set_TRQ0_lo;
	    break;
	case 1:		// 01
	    set_TRQ1_lo;
	    set_TRQ0_hi;
	    break;
	case 2:		// 10
	    set_TRQ1_hi;
	    set_TRQ0_lo;
	    break;
	case 3:		// 11
	    set_TRQ1_hi;
	    set_TRQ0_hi;
	    break;
	}
    }
    if (temp_adecay != G_ADECAY || force_update == true) {
	switch (G_ADECAY) {
	case 0:		// 0
	    enable_ADECAY_output;
	    set_ADECAY_lo;	// DECAY1 OUT
	    enable_ADECAY_input;	// DECAY1 CAP CNTL
	    break;
	case 2:		// 1
	    enable_ADECAY_output;
	    set_ADECAY_hi;	// DECAY1 OUT
	    enable_ADECAY_input;	// DECAY1 CAP CNTL
	    break;
	case 1:		// Z
	    disable_ADECAY_output;
	    set_ADECAY_lo;	// DECAY1 OUT
	    disable_ADECAY_input;	// DECAY1 CAP CNTL
	    break;
	}
	temp_adecay = G_ADECAY;
    }
    // for stepper mode set G_BDECAY = G_ADECAY when ADECAY changes
    if (G_Mode == 2)
	G_BDECAY = G_ADECAY;
    if (temp_bdecay != G_BDECAY || force_update == true) {
	switch (G_BDECAY) {
	case 0:		// 0
	    enable_BDECAY_output;
	    set_BDECAY_lo;	// DECAY1 OUT
	    enable_BDECAY_input;	// DECAY1 CAP CNTL
	    break;
	case 2:		// 1
	    enable_BDECAY_output;
	    set_BDECAY_hi;	// DECAY1 OUT
	    enable_BDECAY_input;	// DECAY1 CAP CNTL
	    break;
	case 1:		// Z
	    disable_BDECAY_output;
	    set_BDECAY_lo;	// DECAY1 OUT
	    disable_BDECAY_input;	// DECAY1 CAP CNTL
	    break;
	}
	temp_bdecay = G_BDECAY;
    }
    if (temp_toff != G_TOFF || force_update == true) {
	// PWM OFF TIME -- 14Apr -- Corrected mismatch between pulldown
	// menu and selection
	switch (G_TOFF) {
	case 0:		// 0, 20us
	    enable_TOFF_output;
	    set_TOFF_lo;	// PWM_OFF_TIME OUT
	    enable_TOFF_input;	// PWM_OFF_TIME CAP CNTL
	    break;
	case 1:		// 1, 30us
	    enable_TOFF_output;
	    set_TOFF_hi;	// PWM_OFF_TIME OUT
	    enable_TOFF_input;	// PWM_OFF_TIME CAP CNTL
	    break;
	case 2:		// Z, 10us
	    disable_TOFF_output;
	    set_TOFF_lo;	// PWM_OFF_TIME OUT
	    disable_TOFF_input;	// PWM_OFF_TIME CAP CNTL
	    break;
	}
	temp_toff = G_TOFF;
    }
    if (temp_G_Mode != G_Mode)	// first select mode configaration
    {
	temp_G_Mode = G_Mode;
	switch (G_Mode) {
	case 0:
	    // normal operation
	    TACTL &= ~ID_3;
	    TBCTL &= ~ID_3;
	    // configure GUI flags for normal operation
	    G_Normal_A = 1;
	    G_Normal_B = 1;
	    G_Parallel = 0;
	    G_Stepper = 0;
	    G_PWM_AIN1 = 0;
	    G_PWM_AIN2 = 0;
	    G_PWM_BIN1 = 0;
	    G_PWM_BIN2 = 0;
	    G_RUN = low;
	    G_DIRECTION = low;
	    // set nSLEEP low to enter sleep mode
	    set_WAKE_lo;
	    set_PARA_lo;
	    // and delay to ensure entry into sleep mode
	    for (i = 10000; i > 0; i--) {
	    };
	    break;		// end of case 0

	case 1:
	    // outputs are placed in parallel
	    TACTL &= ~ID_3;
	    TBCTL &= ~ID_3;
	    G_Normal_A = 1;
	    G_Normal_B = 0;
	    G_Parallel = 1;
	    G_Stepper = 0;
	    G_PWM_AIN1 = 0;
	    G_PWM_AIN2 = 0;
	    G_PWM_BIN1 = 0;
	    G_PWM_BIN2 = 0;
	    G_RUN = low;
	    G_DIRECTION = low;
	    // set nSLEEP low to enter sleep mode
	    set_WAKE_lo;
	    set_PARA_hi;
	    // and delay to ensure entry
	    for (i = 10000; i > 0; i--) {
	    };
	    break;		// end of case 1
	case 2:
	    // stepper operation
	    TACTL |= ID_3;
	    TBCTL |= ID_3;
	    flag = 100;
	    G_Normal_A = 0;
	    G_Normal_B = 0;
	    G_Parallel = 0;
	    G_Stepper = 1;
	    G_PWM_AIN1 = 0;
	    G_PWM_AIN2 = 0;
	    G_PWM_BIN1 = 0;
	    G_PWM_BIN2 = 0;
	    G_RUN = low;
	    G_DIRECTION = low;
	    // set nSLEEP low to enter sleep mode
	    set_WAKE_lo;
	    set_PARA_lo;
	    // and delay to ensure entry
	    for (i = 10000; i > 0; i--) {
	    };
	    break;		// end of case 2
	}
    }				// End of mode selection

    // configure remaining inputs based on mode
    switch (G_Mode) {
    case 0:			// normal mode
	// PWM AIN1, AIN2
	// configure AIN1, AIN2 to drive AOUT1 and AOUT2
	P2SEL |= AIN1 + AIN2;
	P4SEL |= BIN2 + BIN1;
	TACTL |= MC_1 + TASSEL_2 + ID_0;
	TACCTL1 = OUTMOD_7;
	TACCTL2 = OUTMOD_7;
	TACCR0 = 125;
	// adjust duty cycle on AIN1
	if (temp_pwm_ain1 != G_PWM_AIN1 || force_update == true) {
	    temp_pwm_ain1 = G_PWM_AIN1;
	    TACCR1 = (int) (temp_pwm_ain1 * TACCR0 * 0.0101);
	}
	// adjust duty cycle on AIN2
	if (temp_pwm_ain2 != G_PWM_AIN2 || force_update == true) {
	    temp_pwm_ain2 = G_PWM_AIN2;
	    TACCR2 = (int) (temp_pwm_ain2 * TACCR0 * 0.0101);
	}
	// PWM BIN1,BIN2
	// configure AIN1, AIN2 to drive AOUT1 and AOUT2
	TBCTL |= MC_1 + TASSEL_2 + ID_0;
	TBCCTL1 = OUTMOD_7;
	TBCCTL2 = OUTMOD_7;
	TBCCR0 = 125;
	// adjust duty cycle on BIN1
	if (temp_pwm_bin1 != G_PWM_BIN1 || force_update == true) {
	    temp_pwm_bin1 = G_PWM_BIN1;
	    TBCCR2 = (int) (temp_pwm_bin1 * TBCCR0 * 0.0101);
	}
	// adjust duty cycle on BIN2
	if (temp_pwm_bin2 != G_PWM_BIN2 || force_update == true) {
	    temp_pwm_bin2 = G_PWM_BIN2;
	    TBCCR1 = (int) (temp_pwm_bin2 * TBCCR0 * 0.0101);
	}
	break;			// end of case 0, normal operation
    case 1:			// parallel mode (AINx control both
				// outputs)
	P2SEL |= AIN1 + AIN2;
	P4SEL |= BIN2 + BIN1;
	TACTL |= MC_1 + TASSEL_2 + ID_0;
	TACCTL1 = OUTMOD_7;
	TACCTL2 = OUTMOD_7;
	TACCR0 = 125;
	// adjust duty cycle on AIN1
	if (temp_pwm_ain1 != G_PWM_AIN1 || force_update == true) {
	    temp_pwm_ain1 = G_PWM_AIN1;
	    TACCR1 = (int) (temp_pwm_ain1 * TACCR0 * 0.0101);
	}
	// adjust duty cycle on AIN2
	if (temp_pwm_ain2 != G_PWM_AIN2 || force_update == true) {
	    temp_pwm_ain2 = G_PWM_AIN2;
	    TACCR2 = (int) (temp_pwm_ain2 * TACCR0 * 0.0101);
	}
	break;			// end of case 1 parallel mode
    case 2:			// stepper mode
	if (flag != 1) {
	    // initial configuration of timers for stepper operation
	    flag = 1;
	    P2SEL |= AIN1 + AIN2;
	    P4SEL |= BIN2 + BIN1;
	    TACTL &= ~MC_1;
	    TBCTL &= ~MC_1;
	    TACCTL1 &= ~OUTMOD_7;
	    TACCTL2 &= ~OUTMOD_7;
	    TBCCTL1 &= ~OUTMOD_7;
	    TBCCTL2 &= ~OUTMOD_7;
	} else {
	    // re-configure stepper direction and speed if required
	    if (temp_G_DIRECTION != G_DIRECTION
		|| temp_G_PPS_Value != G_PPS_Value) {
		// calculate new speed
		temp_G_PPS_Value = G_PPS_Value;
		Timer_Value = 2000000 / G_PPS_Value;
		// capture direction
		temp_G_DIRECTION = G_DIRECTION;
		TAR = 0;
		TBR = Timer_Value >> 2;
		// set direction and speed
		if (G_DIRECTION == low) {	// configure AINx, BINx to 
						// drive in one
		    // direction
		    P2SEL |= AIN1 + AIN2;
		    P4SEL |= BIN2 + BIN1;
		    // timer A
		    TACCTL1 = OUTMOD_7;
		    TACCTL2 = OUTMOD_3;
		    TACTL |= MC_0 + TASSEL_2 + ID_0;
		    TACCR0 = Timer_Value;	// freq
		    TACCR1 = TACCR2 = Timer_Value >> 1;
		    // timer B
		    TBCTL |= MC_0 + TASSEL_2 + ID_0;
		    TBCCTL1 = OUTMOD_7;
		    TBCCTL2 = OUTMOD_3;
		    TBCCR0 = Timer_Value;	// freq
		    TBCCR1 = TBCCR2 = Timer_Value >> 1;
		}
		if (G_DIRECTION == high) {	// configure AINx, BINx to 
						// drive opposite
		    // direction
		    P2SEL |= AIN1 + AIN2;
		    P4SEL |= BIN2 + BIN1;
		    // timerA
		    TACTL |= MC_0 + TASSEL_2 + ID_0;
		    TACCTL1 = OUTMOD_3;
		    TACCTL2 = OUTMOD_7;
		    TACCR0 = Timer_Value;	// freq
		    TACCR1 = TACCR2 = Timer_Value >> 1;
		    // timeB
		    TBCTL |= MC_0 + TASSEL_2 + ID_0;
		    TBCCTL1 = OUTMOD_7;
		    TBCCTL2 = OUTMOD_3;
		    TBCCR0 = Timer_Value;	// freq
		    TBCCR1 = TBCCR2 = Timer_Value >> 1;
		}
	    }
	}
	// start or stop stepper if change is detected
	if (temp_G_RUN != G_RUN) {
	    temp_G_RUN = G_RUN;
	    if (G_RUN == high) {
		// start stepper
		__bic_SR_register(GIE);
		__no_operation();
		TACTL |= MC_1;
		TBCTL |= MC_1;
		__bis_SR_register(GIE);
		stepper_running = 1;
	    } else {
		// stop stepper
		TACTL &= ~MC_1;
		TBCTL &= ~MC_1;
		TACCTL1 &= ~OUTMOD_7;
		TACCTL2 &= ~OUTMOD_7;
		TBCCTL1 &= ~OUTMOD_7;
		TBCCTL2 &= ~OUTMOD_7;
		stepper_running = 0;
		temp_G_DIRECTION = 100;
		temp_G_PPS_Value = 12000;
	    }
	}
	break;			// end of case 2, stepper mode
    }
    // From Micro to GUI
    // read nFAULT and send status
    if (read_FAULT_pin == 0)
	G_FAULT = low;
    else
	G_FAULT = high;
    // Set DACs value by multiplying by 4
    DAC12_0DAT = G_AVREF_VAL << 2;
    // if in stepper mode set both DACs to the same value
    if (G_Stepper == 1)
	DAC12_1DAT = G_AVREF_VAL << 2;
    else
	DAC12_1DAT = G_BVREF_VAL << 2;
    // update the full scale chopping current if necessary
    if ((temp_torque_val != G_TORQUE) || force_update == true) {
	temp_torque_val = G_TORQUE;
	torque_multiplier = (4 - temp_torque_val) * TORQUE_STEP;
	UpdateChoppingCurrentA = true;
	UpdateChoppingCurrentB = true;
    }
    if ((temp_avref_val != G_AVREF_VAL) || force_update == true) {
	temp_avref_val = G_AVREF_VAL;
	avref_multiplier = temp_avref_val * VREF_BIT;	// vref bit =
							// 1/1023
	UpdateChoppingCurrentA = true;
    }
    if ((temp_bvref_val != G_BVREF_VAL) || force_update == true) {
	temp_bvref_val = G_BVREF_VAL;
	bvref_multiplier = temp_bvref_val * VREF_BIT;	// vref bit =
							// 1/1023
	UpdateChoppingCurrentB = true;
    }
    if ((temp_G_IRSENSE != G_IRSENSE) || force_update == true) {
	// adjust sense resistor value in half if using parallel mode
	if (G_Parallel == 1)
	    temp_G_IRSENSE = G_IRSENSE * 0.5;
	else
	    temp_G_IRSENSE = G_IRSENSE;
	irsense_multiplier = 1 / temp_G_IRSENSE;	// Could not avoid 
							// division here,
							// but it is the
							// only place
	UpdateChoppingCurrentA = true;
	UpdateChoppingCurrentB = true;
    }
    if (UpdateChoppingCurrentA == true || UpdateChoppingCurrentB == true)	// recalculate 
										// chopping 
										// current 
										// based 
										// on 
										// multipliers
    {
	// multiply common numbers here to avoid duplicate multiplies
	vref_times_irsense_times_constant =
	    V3P3_TIMES_GAIN * irsense_multiplier * torque_multiplier;
	if (UpdateChoppingCurrentA == true)
	    G_AIFS = avref_multiplier * vref_times_irsense_times_constant;
	if (UpdateChoppingCurrentB == true & G_Normal_B == 1)
	    G_BIFS = bvref_multiplier * vref_times_irsense_times_constant;
	// reset flags
	UpdateChoppingCurrentA = false;
	UpdateChoppingCurrentA = false;
	force_update = false;
    }
}				// end of UpdateSystem

void
SetDefaultGUIVals()
{
    G_FIRMWARE_VERSION = FW_VERSION;
    G_AIFS = DEFAULT_IFS;
    G_BIFS = DEFAULT_IFS;
    G_IRSENSE = DEFAULT_SENSE_RESISTOR;
    G_AVREF_VAL = DEFAULT_VREF_VAL;
    G_BVREF_VAL = DEFAULT_VREF_VAL;
    G_PPS_Value = 200;		// 200 stepper pulses per second
    G_TORQUE = DEFAULT_TORQUE;
    G_ADECAY = DEFAULT_DECAY;
    G_BDECAY = DEFAULT_DECAY;
}

void
SetDefaultGPIOVals()
{
    G_WAKE = low;
    G_FAULT = high;
}
